<?php

namespace App\Http\Controllers;

use App\Models\OfertaLaboral;
use App\Models\Empresa;
use App\Models\PlanEstudio;
use App\Models\TipoContrato;
use App\Models\Postulacion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

use App\Models\Instituto;
class OfertaLaboralController extends Controller
{
    public function index()
    {
        $ofertas = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])->get();
        $empresas = Empresa::all();
        $planes_estudio = PlanEstudio::all();
        $tipos_contrato = TipoContrato::all();

        return view('ofertas_laborales.index', compact('ofertas', 'empresas', 'planes_estudio', 'tipos_contrato'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'fecha_publicacion' => 'required|date',
            'fecha_limite' => 'required|date|after_or_equal:fecha_publicacion',
            'id_tipo_contrato' => 'nullable|exists:tipos_contrato,id_tipo_contrato',
            'id_empresa' => 'nullable|exists:empresas,id_empresa',
            'id_plan_estudio' => 'nullable|exists:planes_estudio,id',
            'lugar' => 'nullable|string|max:100',
            'tipo_oferta' => 'nullable|string|max:100',
            'remuneracion' => 'nullable|numeric',
            'vacantes' => 'nullable|string|max:3',
            'experiencia' => 'nullable|string',
            'pasos_postular' => 'nullable|string',
            'estado' => 'required|in:aceptado,pendiente,rechazado',
            'archivo_pdf' => 'nullable|file|mimes:pdf|max:2048',
        ]);

        $data = $request->except('archivo_pdf');

        if ($request->hasFile('archivo_pdf')) {
            $file = $request->file('archivo_pdf');
            $filename = uniqid('oferta_') . '.' . $file->getClientOriginalExtension();

            $destinationPath = public_path('ofertas_pdf');

            // Crear carpeta si no existe
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            $file->move($destinationPath, $filename);
            $data['archivo_pdf'] = $filename;
        }

        OfertaLaboral::create($data);

        return response()->json(['message' => 'Oferta laboral creada correctamente']);
    }


    public function update(Request $request, $id)
    {
        $oferta = OfertaLaboral::findOrFail($id);

        $request->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'fecha_publicacion' => 'required|date',
            'fecha_limite' => 'required|date|after_or_equal:fecha_publicacion',
            'id_tipo_contrato' => 'nullable|exists:tipos_contrato,id_tipo_contrato',
            'id_empresa' => 'nullable|exists:empresas,id_empresa',
            'id_plan_estudio' => 'nullable|exists:planes_estudio,id',
            'lugar' => 'nullable|string|max:100',
            'tipo_oferta' => 'nullable|string|max:100',
            'remuneracion' => 'nullable|numeric',
            'vacantes' => 'nullable|string|max:3',
            'experiencia' => 'nullable|string',
            'pasos_postular' => 'nullable|string',
            'estado' => 'required|in:aceptado,pendiente,rechazado',
            'archivo_pdf' => 'nullable|file|mimes:pdf|max:2048',
        ]);

        $data = $request->except('archivo_pdf');

        if ($request->hasFile('archivo_pdf')) {
            // Eliminar el anterior si existe
            if ($oferta->archivo_pdf && file_exists(public_path('ofertas_pdf/' . $oferta->archivo_pdf))) {
                unlink(public_path('ofertas_pdf/' . $oferta->archivo_pdf));
            }

            $file = $request->file('archivo_pdf');
            $filename = uniqid('oferta_') . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('ofertas_pdf'), $filename);
            $data['archivo_pdf'] = $filename;
        }

        $oferta->update($data);

        return response()->json(['message' => 'Oferta laboral actualizada correctamente']);
    }

    public function destroy($id)
    {
        $oferta = OfertaLaboral::findOrFail($id);

        if ($oferta->archivo_pdf && file_exists(public_path('ofertas_pdf/' . $oferta->archivo_pdf))) {
            unlink(public_path('ofertas_pdf/' . $oferta->archivo_pdf));
        }

        $oferta->delete();

        return response()->json(['message' => 'Oferta laboral eliminada correctamente']);
    }

    public function show($id)
    {
        $oferta = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])->findOrFail($id);
        return response()->json($oferta);
    }

    public function verPdf($id)
    {
        $oferta = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])->findOrFail($id);

        // Asegúrate de tener el archivo: resources/views/ofertas_laborales/pdf.blade.php
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('ofertas_laborales.pdf', compact('oferta'));

        return $pdf->stream("oferta_{$oferta->id_oferta}.pdf"); // Muestra el PDF en una nueva pestaña
    }

    public function reporteVista()
{
    $planes = PlanEstudio::all();
    return view('ofertas_laborales.reporte', compact('planes'));
}

public function reporteAjax(Request $request)
{
    $query = OfertaLaboral::with(['empresa', 'planEstudio', 'tipoContrato']);

    if ($request->filled('desde')) {
        $query->where('fecha_publicacion', '>=', $request->desde);
    }

    if ($request->filled('hasta')) {
        $query->where('fecha_publicacion', '<=', $request->hasta);
    }

    if ($request->filled('id_plan_estudio')) {
        $query->where('id_plan_estudio', $request->id_plan_estudio);
    }

    $ofertas = $query->get();

    return view('ofertas_laborales.partials.tabla_resultados', compact('ofertas'))->render();
}

public function verPostulantes($id)
{
    $oferta = OfertaLaboral::with(['empresa', 'planEstudio', 'tipoContrato'])->findOrFail($id);

    $postulaciones = Postulacion::with('postulante')
        ->where('id_oferta', $id)
        ->get();

    return view('ofertas_laborales.postulantes', compact('oferta', 'postulaciones'));
}

public function listarVencidas()
{
    $fechaHoy = now()->toDateString();

    $ofertasVencidas = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])
        ->where('fecha_limite', '<', $fechaHoy)
        ->get();

    return view('ofertas_laborales.vencidas', compact('ofertasVencidas', 'fechaHoy'));
}

public function reporteVencidasPdf(Request $request)
{
    $request->validate([
        'fecha_inicio' => 'required|date',
        'fecha_fin' => 'required|date|after_or_equal:fecha_inicio',
    ]);

    $fechaInicio = $request->fecha_inicio;
    $fechaFin = $request->fecha_fin;

    // Ofertas vencidas dentro del rango de fechas
    $ofertasVencidas = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])
        ->where('fecha_limite', '<', now())
        ->whereBetween('fecha_limite', [$fechaInicio, $fechaFin])
        ->get();

    $instituto = Instituto::first(); // Se asume que hay uno

    $pdf = PDF::loadView('ofertas_laborales.reporte_vencidas_pdf', compact('ofertasVencidas', 'instituto', 'fechaInicio', 'fechaFin'))
        ->setPaper('a4', 'landscape');

    return $pdf->stream('reporte_ofertas_vencidas.pdf');
}

}
