<?php

namespace App\Http\Controllers;

use App\Models\Periodo;
use Illuminate\Http\Request;

class PeriodoController extends Controller
{
    /**
     * Mostrar una lista de los periodos.
     */
    public function index()
    {
        $periodos = Periodo::all(); // Obtener todos los periodos
        return view('periodos.index', compact('periodos')); // Retornar la vista con los periodos
    }

    /**
     * Mostrar el formulario para crear un nuevo periodo.
     */
    public function create()
    {
        return view('periodos.create'); // Retornar vista para crear un nuevo periodo
    }

    /**
     * Almacenar un nuevo periodo en la base de datos.
     */
    public function store(Request $request)
    {
        // Validar los datos recibidos
        $request->validate([
            'nombre' => 'required|string|max:50|unique:periodos,nombre',
            'descripcion' => 'nullable|string',
            'fecha_inicio' => 'required|date',
            'fecha_fin' => 'required|date|after:fecha_inicio',
        ]);

        // Crear el nuevo periodo
        $periodo = Periodo::create($request->all());

        // Si la petición es AJAX, devolver JSON
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Periodo creado exitosamente.',
                'periodo' => $periodo
            ]);
        }

        // Si la petición NO es AJAX, redirigir con un mensaje de éxito
        return redirect()->route('periodos.index')->with('success', 'Periodo creado exitosamente.');
    }


    /**
     * Mostrar el periodo especificado.
     */
    public function show($id)
    {
        $periodo = Periodo::findOrFail($id); // Obtener el periodo por ID
        return view('periodos.show', compact('periodo')); // Retornar vista del periodo
    }

    /**
     * Mostrar el formulario para editar un periodo.
     */
    public function edit($id)
    {
        $periodo = Periodo::findOrFail($id); // Obtener el periodo a editar
        return view('periodos.edit', compact('periodo')); // Retornar vista de edición
    }

    /**
     * Actualizar el periodo especificado en la base de datos.
     */
    public function update(Request $request, $id)
    {
        // Validar los datos
        $request->validate([
            'nombre' => 'required|string|max:50|unique:periodos,nombre,' . $id,
            'descripcion' => 'nullable|string',
            'fecha_inicio' => 'required|date',
            'fecha_fin' => 'required|date|after:fecha_inicio',
        ]);

        // Buscar el periodo por ID y actualizar los valores
        $periodo = Periodo::findOrFail($id);
        $periodo->update($request->all());

        // Redirigir de vuelta con mensaje de éxito
        return redirect()->route('periodos.index')->with('success', 'Periodo actualizado exitosamente.');
    }



    /**
     * Eliminar el periodo especificado de la base de datos.
     */
    public function destroy($id)
    {
        // Encontrar y eliminar el periodo
        $periodo = Periodo::findOrFail($id);
        $periodo->delete();

        // Redirigir a la lista de periodos con un mensaje de éxito
        return redirect()->route('periodos.index')->with('success', 'Periodo eliminado exitosamente.');
    }
}
